// Quotation Application JavaScript

document.addEventListener('DOMContentLoaded', function() {
    // Initialize the application
    initializeNavigation();
    initializePrintOptimization();
    initializeScrollEffects();
    
    console.log('Drupal Migration Quotation App initialized');
});

// Navigation Functions
function initializeNavigation() {
    // Smooth scroll for navigation links
    const navLinks = document.querySelectorAll('.nav-links a');
    
    navLinks.forEach(link => {
        link.addEventListener('click', function(e) {
            e.preventDefault();
            const targetId = this.getAttribute('href');
            const targetElement = document.querySelector(targetId);
            
            if (targetElement) {
                const navHeight = document.querySelector('.nav-menu').offsetHeight;
                const targetPosition = targetElement.offsetTop - navHeight - 20;
                
                window.scrollTo({
                    top: targetPosition,
                    behavior: 'smooth'
                });
                
                // Update active nav item
                updateActiveNavItem(this);
            }
        });
    });
    
    // Update active navigation item on scroll
    window.addEventListener('scroll', debounce(updateNavOnScroll, 100));
}

function updateActiveNavItem(activeLink) {
    const navLinks = document.querySelectorAll('.nav-links a');
    navLinks.forEach(link => link.classList.remove('active'));
    activeLink.classList.add('active');
}

function updateNavOnScroll() {
    const sections = document.querySelectorAll('[id]');
    const navHeight = document.querySelector('.nav-menu').offsetHeight;
    const scrollPosition = window.scrollY + navHeight + 100;
    
    let currentSection = '';
    
    sections.forEach(section => {
        const sectionTop = section.offsetTop;
        const sectionHeight = section.offsetHeight;
        
        if (scrollPosition >= sectionTop && scrollPosition < sectionTop + sectionHeight) {
            currentSection = section.getAttribute('id');
        }
    });
    
    if (currentSection) {
        const activeLink = document.querySelector(`.nav-links a[href="#${currentSection}"]`);
        if (activeLink) {
            updateActiveNavItem(activeLink);
        }
    }
}

// Phase Toggle Functions
function togglePhase(phaseId) {
    const phaseDetails = document.getElementById(phaseId);
    const phaseItem = phaseDetails.closest('.phase-item');
    const toggleIcon = phaseItem.querySelector('.toggle-icon');
    
    if (phaseDetails.classList.contains('visible')) {
        // Collapse
        phaseDetails.classList.remove('visible');
        phaseItem.classList.remove('expanded');
        toggleIcon.textContent = '+';
        
        // Smooth collapse
        phaseDetails.style.maxHeight = phaseDetails.scrollHeight + 'px';
        setTimeout(() => {
            phaseDetails.style.maxHeight = '0';
        }, 10);
        
        setTimeout(() => {
            phaseDetails.style.display = 'none';
            phaseDetails.style.maxHeight = '';
        }, 300);
    } else {
        // Expand
        phaseDetails.style.display = 'block';
        phaseDetails.classList.add('visible');
        phaseItem.classList.add('expanded');
        toggleIcon.textContent = '−';
        
        // Smooth expand
        phaseDetails.style.maxHeight = '0';
        setTimeout(() => {
            phaseDetails.style.maxHeight = phaseDetails.scrollHeight + 'px';
        }, 10);
        
        setTimeout(() => {
            phaseDetails.style.maxHeight = '';
        }, 300);
    }
}

// Print Optimization
function initializePrintOptimization() {
    // Expand all phases for printing
    window.addEventListener('beforeprint', function() {
        const phaseDetails = document.querySelectorAll('.phase-details');
        phaseDetails.forEach(detail => {
            detail.style.display = 'block';
            detail.classList.add('visible');
        });
        
        const phaseItems = document.querySelectorAll('.phase-item');
        phaseItems.forEach(item => {
            item.classList.add('expanded');
        });
        
        const toggleIcons = document.querySelectorAll('.toggle-icon');
        toggleIcons.forEach(icon => {
            icon.textContent = '−';
        });
        
        console.log('Print mode: All sections expanded');
    });
    
    // Restore original state after printing
    window.addEventListener('afterprint', function() {
        const phaseDetails = document.querySelectorAll('.phase-details');
        phaseDetails.forEach(detail => {
            if (!detail.classList.contains('user-expanded')) {
                detail.style.display = 'none';
                detail.classList.remove('visible');
            }
        });
        
        const phaseItems = document.querySelectorAll('.phase-item');
        phaseItems.forEach(item => {
            if (!item.classList.contains('user-expanded')) {
                item.classList.remove('expanded');
            }
        });
        
        const toggleIcons = document.querySelectorAll('.toggle-icon');
        toggleIcons.forEach(icon => {
            const phaseItem = icon.closest('.phase-item');
            if (!phaseItem.classList.contains('expanded')) {
                icon.textContent = '+';
            }
        });
        
        console.log('Print mode: Restored original state');
    });
}

// Scroll Effects
function initializeScrollEffects() {
    // Add scroll-based animations
    const observerOptions = {
        threshold: 0.1,
        rootMargin: '0px 0px -50px 0px'
    };
    
    const observer = new IntersectionObserver(function(entries) {
        entries.forEach(entry => {
            if (entry.isIntersecting) {
                entry.target.classList.add('animate-in');
            }
        });
    }, observerOptions);
    
    // Observe sections for animation
    const sections = document.querySelectorAll('.section, .quote-header, .client-info');
    sections.forEach(section => {
        observer.observe(section);
    });
}

// Utility Functions
function debounce(func, wait) {
    let timeout;
    return function executedFunction(...args) {
        const later = () => {
            clearTimeout(timeout);
            func(...args);
        };
        clearTimeout(timeout);
        timeout = setTimeout(later, wait);
    };
}

// Cost Calculator Functions
function calculateTotalCost() {
    const phases = [
        { name: 'Discovery & Analysis', cost: 218032 },
        { name: 'Environment Setup', cost: 94394 },
        { name: 'Content & Data Migration', cost: 271924 },
        { name: 'Frontend Theme Development', cost: 538252 },
        { name: 'Backend Development', cost: 817384 },
        { name: 'Testing & Quality Assurance', cost: 334136 },
        { name: 'Launch & Training', cost: 145488 }
    ];
    
    const total = phases.reduce((sum, phase) => sum + phase.cost, 0);
    return {
        phases: phases,
        total: total,
        formatted: formatCurrency(total)
    };
}

function formatCurrency(amount) {
    return new Intl.NumberFormat('en-IN', {
        style: 'currency',
        currency: 'INR',
        minimumFractionDigits: 0,
        maximumFractionDigits: 0
    }).format(amount);
}

// Package Comparison Functions
function highlightRecommendedPackage() {
    const recommendedCard = document.querySelector('.package-card.recommended');
    if (recommendedCard) {
        recommendedCard.style.transform = 'scale(1.02)';
        setTimeout(() => {
            recommendedCard.style.transform = '';
        }, 2000);
    }
}

// Interactive Features
function addInteractiveFeatures() {
    // Add hover effects to phase items
    const phaseHeaders = document.querySelectorAll('.phase-header');
    phaseHeaders.forEach(header => {
        header.addEventListener('mouseenter', function() {
            this.style.transform = 'translateX(5px)';
        });
        
        header.addEventListener('mouseleave', function() {
            this.style.transform = '';
        });
    });
    
    // Add click effects to package cards
    const packageCards = document.querySelectorAll('.package-card');
    packageCards.forEach(card => {
        card.addEventListener('click', function() {
            // Remove selection from other cards
            packageCards.forEach(c => c.classList.remove('selected'));
            
            // Add selection to clicked card
            this.classList.add('selected');
            
            // Log selection
            const packageName = this.querySelector('h3').textContent;
            console.log(`Package selected: ${packageName}`);
        });
    });
}

// Initialize interactive features after DOM load
document.addEventListener('DOMContentLoaded', function() {
    setTimeout(() => {
        addInteractiveFeatures();
        highlightRecommendedPackage();
    }, 1000);
});

// Navigation Menu Mobile Toggle (if needed)
function initializeMobileMenu() {
    const navContent = document.querySelector('.nav-content');
    const navLinks = document.querySelector('.nav-links');
    
    // Check if mobile menu is needed
    if (window.innerWidth <= 768) {
        // Add mobile menu functionality if needed
        console.log('Mobile view detected');
    }
}

window.addEventListener('resize', debounce(initializeMobileMenu, 250));

// Analytics and Tracking Functions
function trackUserInteraction(action, element) {
    // Log user interactions for analytics
    console.log(`User interaction: ${action} on ${element}`);
    
    // In a real application, you would send this data to an analytics service
    // Example: gtag('event', action, { element: element });
}

// Add tracking to important elements
document.addEventListener('DOMContentLoaded', function() {
    // Track phase expansions
    const phaseHeaders = document.querySelectorAll('.phase-header');
    phaseHeaders.forEach((header, index) => {
        header.addEventListener('click', function() {
            const phaseName = this.querySelector('h3').textContent;
            trackUserInteraction('phase_toggle', phaseName);
        });
    });
    
    // Track package selections
    const packageCards = document.querySelectorAll('.package-card');
    packageCards.forEach(card => {
        card.addEventListener('click', function() {
            const packageName = this.querySelector('h3').textContent;
            trackUserInteraction('package_click', packageName);
        });
    });
    
    // Track print button clicks
    const printButtons = document.querySelectorAll('[onclick*="print"]');
    printButtons.forEach(button => {
        button.addEventListener('click', function() {
            trackUserInteraction('print_quote', 'print_button');
        });
    });
});

// Accessibility Enhancements
function initializeAccessibility() {
    // Add keyboard navigation for phase toggles
    const phaseHeaders = document.querySelectorAll('.phase-header');
    phaseHeaders.forEach(header => {
        header.setAttribute('tabindex', '0');
        header.setAttribute('role', 'button');
        header.setAttribute('aria-expanded', 'false');
        
        header.addEventListener('keydown', function(e) {
            if (e.key === 'Enter' || e.key === ' ') {
                e.preventDefault();
                this.click();
            }
        });
    });
    
    // Update aria-expanded attributes when phases are toggled
    window.togglePhase = function(phaseId) {
        const phaseDetails = document.getElementById(phaseId);
        const phaseItem = phaseDetails.closest('.phase-item');
        const phaseHeader = phaseItem.querySelector('.phase-header');
        const toggleIcon = phaseItem.querySelector('.toggle-icon');
        
        if (phaseDetails.classList.contains('visible')) {
            // Collapse
            phaseDetails.classList.remove('visible');
            phaseItem.classList.remove('expanded');
            toggleIcon.textContent = '+';
            phaseHeader.setAttribute('aria-expanded', 'false');
            
            // Smooth collapse
            phaseDetails.style.maxHeight = phaseDetails.scrollHeight + 'px';
            setTimeout(() => {
                phaseDetails.style.maxHeight = '0';
            }, 10);
            
            setTimeout(() => {
                phaseDetails.style.display = 'none';
                phaseDetails.style.maxHeight = '';
            }, 300);
        } else {
            // Expand
            phaseDetails.style.display = 'block';
            phaseDetails.classList.add('visible');
            phaseItem.classList.add('expanded');
            toggleIcon.textContent = '−';
            phaseHeader.setAttribute('aria-expanded', 'true');
            
            // Smooth expand
            phaseDetails.style.maxHeight = '0';
            setTimeout(() => {
                phaseDetails.style.maxHeight = phaseDetails.scrollHeight + 'px';
            }, 10);
            
            setTimeout(() => {
                phaseDetails.style.maxHeight = '';
            }, 300);
        }
    };
}

// Initialize accessibility features
document.addEventListener('DOMContentLoaded', function() {
    initializeAccessibility();
});

// Error Handling
window.addEventListener('error', function(e) {
    console.error('Application error:', e.error);
    // In production, you might want to report this to an error tracking service
});

// Performance Monitoring
function measurePerformance() {
    if ('performance' in window) {
        window.addEventListener('load', function() {
            setTimeout(() => {
                const perfData = performance.getEntriesByType('navigation')[0];
                console.log('Page load time:', perfData.loadEventEnd - perfData.loadEventStart + 'ms');
            }, 0);
        });
    }
}

measurePerformance();